import Image from "next/image";
import { useState, useMemo, useEffect } from "react";
import Lightbox from "yet-another-react-lightbox";
import "yet-another-react-lightbox/styles.css";

export default function Gallery({ images = [], url_section_image, url_base }) {
  if (images.length === 0) return null;

  const [index, setIndex] = useState(-1);
  const [windowWidth, setWindowWidth] = useState(1200);
  const [visibleCount, setVisibleCount] = useState(9);

  useEffect(() => {
    function handleResize() {
      setWindowWidth(window.innerWidth);
    }
    handleResize();
    window.addEventListener("resize", handleResize);
    return () => window.removeEventListener("resize", handleResize);
  }, []);

  const gridCombinationsPar = [
    ["col-xl-4", "col-xl-4", "col-xl-4"],
    ["col-xl-3", "col-xl-3", "col-xl-6"],
    ["col-xl-5", "col-xl-4", "col-xl-3"],
    ["col-xl-6", "col-xl-3", "col-xl-3"],
    ["col-xl-3", "col-xl-3", "col-xl-6"],
    ["col-xl-2", "col-xl-4", "col-xl-6"],
    ["col-xl-3", "col-xl-6", "col-xl-3"],
    ["col-xl-4", "col-xl-5", "col-xl-3"],
    ["col-xl-5", "col-xl-3", "col-xl-4"],
  ];

  const gridCombinationsImpar = [
    ["col-xl-6", "col-xl-6"],
    ["col-xl-4", "col-xl-8"],
    ["col-xl-8", "col-xl-4"],
    ["col-xl-5", "col-xl-7"],
    ["col-xl-7", "col-xl-5"],
  ];

  const gridCombinationsComplete = [["col-xl-12"]];

  const grouped = useMemo(() => {
    const result = [];
    for (let i = 0; i < images.length; i += 3) {
      result.push(images.slice(i, i + 3));
    }
    return result;
  }, [images]);

  const allImages = grouped.flat();
  const visibleImagesFlat = allImages.slice(0, visibleCount);
  const visibleGrouped = [];
  for (let i = 0; i < visibleImagesFlat.length; i += 3) {
    visibleGrouped.push(visibleImagesFlat.slice(i, i + 3));
  }

  const gridClasses = useMemo(() => {
    return grouped.map((group, i) => {
      if (group.length === 1) {
        return gridCombinationsComplete[
          Math.floor(Math.random() * gridCombinationsComplete.length)
        ];
      } else if (group.length === 2 && i === grouped.length - 1) {
        return gridCombinationsImpar[
          Math.floor(Math.random() * gridCombinationsImpar.length)
        ];
      } else {
        return gridCombinationsPar[
          Math.floor(Math.random() * gridCombinationsPar.length)
        ];
      }
    });
  }, [grouped]);

  function getSizesFromClass(colClass) {
    if (!colClass) return "(max-width: 768px) 100vw, 100vw";

    const match = colClass.match(/col-(?:xs|sm|md|lg|xl|xxl)-(\d{1,2})/);
    const col = match ? parseInt(match[1], 10) : 12;

    if (windowWidth < 768) return "(max-width: 768px) 100vw, 100vw";

    const pct = Math.max(1, Math.min(100, (col / 12) * 100));
    return `(max-width: 768px) 100vw, ${pct.toFixed(2)}vw`;
  }

  function pickVariantDynamic(imageUrl, colClass = null) {
    let sizeSuffix;

    if (windowWidth < 768) {
      sizeSuffix = "_medium";
    } else if (colClass) {
      const match = colClass.match(/col-(?:xs|sm|md|lg|xl|xxl)-(\d{1,2})/);
      const col = match ? parseInt(match[1], 10) : 12;

      if (col <= 2) {
        sizeSuffix = "_thumb";
      } else if (col <= 4) {
        sizeSuffix = "_medium";
      } else {
        sizeSuffix = "_large";
      }
    } else {
      sizeSuffix = "_large";
    }

    const ext = imageUrl.substring(imageUrl.lastIndexOf("."));
    const base = imageUrl.substring(0, imageUrl.lastIndexOf("."));
    const pathVariant = `${base}${sizeSuffix}${ext}`;

    return pathVariant;
  }

  function pickLightboxVariant(imageUrl) {
    const width = windowWidth;
    let sizeSuffix;

    if (width <= 480) {
      sizeSuffix = "_medium";
    } else {
      sizeSuffix = "_large";
    }

    const ext = imageUrl.substring(imageUrl.lastIndexOf("."));
    const base = imageUrl.substring(0, imageUrl.lastIndexOf("."));

    const pathVariant = `${base}${sizeSuffix}${ext}`;

    return pathVariant;
  }

  return (
    <>
      <div className="gallery_area" id="gallery">
        <div className="container">
          <div className="row">
            <div className="col-xl-12">
              <div className="section_title text-center">
                <img src={url_section_image} alt="" />
                <h3>Galeria de fotos</h3>
              </div>
            </div>
          </div>
        </div>

        <div className="container-fluid p-0">
          <div className="row grid no-gutters">
            {visibleGrouped.map((group, groupIndex) =>
              group.map((image, index) => (
                <div
                  key={`${groupIndex}-${index}`}
                  className={`${gridClasses[groupIndex][index]} col-md-6 col-lg-6 grid-item`}
                >
                  <div
                    className="single_gallery"
                    onClick={() => setIndex(groupIndex * 3 + index)}
                  >
                    <Image
                      src={`${url_base}/${pickVariantDynamic(
                        image.url,
                        gridClasses[groupIndex][index]
                      )}`}
                      alt={`Imagen ${index}`}
                      fill
                      style={{ objectFit: "cover" }}
                      sizes={getSizesFromClass(gridClasses[groupIndex][index])}
                      priority={groupIndex === 0 && index === 0}
                    />
                  </div>
                </div>
              ))
            )}
          </div>
          {visibleCount < allImages.length && (
            <div className="text-center my-4">
              <button
              className="btn btn-success fs-4 p-2 px-5"
                onClick={() => setVisibleCount((prev) => prev + 9)}
              >
                Mostrar más recuerdos
              </button>
            </div>
          )}
        </div>

        <Lightbox
          open={index >= 0}
          close={() => setIndex(-1)}
          index={index}
          slides={images.map((image) => ({
            src: `${url_base}/${pickLightboxVariant(image.url)}`,
          }))}
          animation={{ fade: 250 }}
          styles={{ container: { backgroundColor: "rgba(0,0,0,0.7)" } }}
        />
      </div>
    </>
  );
}
