"use client";

import { useEffect, useRef, useState } from "react";
import Layout from "@/app/layout/layout-intranet";
import DataTable from "react-data-table-component";
import Loader from "@/components/Loader";
import Link from "next/link";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { faEdit, faPaperPlane } from "@fortawesome/free-solid-svg-icons";
import { api } from "@/libs/apiClient";
import { faWhatsapp } from "@fortawesome/free-brands-svg-icons";
import Modal from "@/components/Modal";

export default function Usuarios() {
  const [data, setData] = useState([]);
  const [language, setLanguage] = useState(null);
  const [loading, setLoading] = useState(true);
  const [filterText, setFilterText] = useState("");
  const [modalMsg, setmodalMsg] = useState("");
  const [filteredItems, setFilteredItems] = useState([]);
  const tableRef = useRef(null);
  const [showModal, setShowModal] = useState(false);
  const [typeModal, setTypeModal] = useState("");

  useEffect(() => {
    const fetchData = async () => {
      try {
        const data = await api.get("/usuarios");
        setData(data);
      } catch (error) {
        console.error("Error al cargar los usuarios:", error.message);
      } finally {
        setLoading(false);
      }
    };

    fetchData();
  }, []);

  useEffect(() => {
    fetch("/assets/lang/DataTables-ES.json")
      .then((res) => res.json())
      .then((data) => {
        setLanguage(data);
        setLoading(false);
      })
      .catch((error) => console.error("Error al cargar el idioma:", error));
  }, [data]);

  const columns = [
    {
      name: "",
      selector: (row) => row.id,
      sortable: true,
      cell: (row) => (
        <Link href={`/usuarios/${row.id}`}>
          <button className="px-2 py-1 btn btn-warning">
            <FontAwesomeIcon icon={faEdit}></FontAwesomeIcon>
          </button>
        </Link>
      ),
      ignoreRowClick: true,
      allowOverFlow: true,
      button: "true",
    },
    {
      name: "Usuario",
      selector: (row) => row.usuario,
      sortable: true,
    },
    {
      name: "Nombres",
      selector: (row) => row.nombres,
      sortable: true,
    },
    {
      name: "Telefono",
      selector: (row) => row.telefono,
      sortable: true,
    },
    {
      name: "Email",
      selector: (row) => row.email,
      sortable: true,
    },
    {
      name: "",
      selector: (row) => row.id,
      sortable: true,
      cell: (row) => (
        <button
          className="px-2 py-1 btn btn-success"
          type="button"
          onClick={() => handleSendWhatsApp(row.id)}
        >
          <FontAwesomeIcon icon={faWhatsapp}></FontAwesomeIcon>{" "}
          <FontAwesomeIcon icon={faPaperPlane}></FontAwesomeIcon>
        </button>
      ),
      ignoreRowClick: true,
      allowOverFlow: true,
      button: "true",
    },
  ];

  const handleSendWhatsApp = async (id) => {
    try {
        const {total, enviados, fallidos} = await api.post("/whatsapp/sendmsg", { id: id });

    if (total === 0) {
    setmodalMsg("No hay invitados para enviar mensajes.");
    setTypeModal("info");
  } else if (enviados === total) {
    setmodalMsg("¡Mensajes de WhatsApp enviados con éxito!");
    setTypeModal("success");
  } else if (enviados > 0 && fallidos > 0) {
    setmodalMsg("Precaución: algunos mensajes de WhatsApp no se enviaron, revisar logs.");
    setTypeModal("warning");
  } else if (enviados === 0 && fallidos === total) {
    setmodalMsg("Error: no se pudo enviar ningún mensaje de WhatsApp, revisar logs.");
    setTypeModal("danger");
  } else {
    setmodalMsg("Resultado desconocido al enviar los mensajes de WhatsApp, revisar logs.");
    setTypeModal("danger");
  }

  setShowModal(true);
    
    } catch (err) {
      setmodalMsg("Error al cargar los usuarios");
      setTypeModal("danger");
      setShowModal(true);
      console.error("error en invitados list", err);
    }
  };

  const handleFilter = (event) => {
    const filterValue = event.target.value;
    setFilterText(filterValue);

    const filteredData = data.filter((item) => {
      return Object.values(item)
        .join(" ")
        .toLowerCase()
        .includes(filterValue.toLowerCase());
    });

    if (filteredData.length === 0) {
      setFilteredItems([
        {
          mensaje: "No hay registro",
        },
      ]);
    } else {
      setFilteredItems(filteredData);
    }
  };

  return (
    <Layout>
      {data && loading ? (
        <Loader />
      ) : (
        <>
          <div className="container mt-6">
            <h1 className="mb-4">Usuarios</h1>
            <section className="mb-5">
              <div className="row mt-5">
                <div className="col-md-8 mb-3"></div>
                <div className="col-md-4 mb-3">
                  <input
                    type="text"
                    className="alert alert-warning form-control p-2 "
                    placeholder="Buscar"
                    value={filterText}
                    onChange={handleFilter}
                  />
                </div>
              </div>
              <div ref={tableRef} className="col-md-12">
                <DataTable
                  columns={columns}
                  data={filteredItems.length > 0 ? filteredItems : data}
                  pagination
                  highlightOnHover
                  responsive
                  progressPending={data.length === 0}
                  paginationComponentOptions={language || {}}
                  id="dt_invitados"
                />
              </div>
            </section>
          </div>
          {showModal && (
            <Modal
              onClose={() => setShowModal(false)}
              title="Mensaje"
              type={typeModal}
            >
              <p>{modalMsg}</p>
            </Modal>
          )}
        </>
      )}
    </Layout>
  );
}
