"use client";

import React, { useEffect, useState, useRef } from "react";
import { useRouter } from "next/navigation";
import Layout from "@/app/layout/layout-intranet";
import Loader from "@/components/Loader";
import DataTable from "react-data-table-component";
import ExcelJS from "exceljs";
import jsPDF from "jspdf";
import autoTable from "jspdf-autotable";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import "@/styles/custom-panel.css";
import {
  faFileExcel,
  faFilePdf,
  faPrint,
} from "@fortawesome/free-solid-svg-icons";
import { api } from "@/libs/apiClient";
import { getSession } from "@/libs/session";
import Mantenedor_Invitados from "@/components/mantenedor/Mantenedor_Invitados";

export default function Panel() {
  const [session, setSession] = useState(undefined);
  const [data, setData] = useState([]);
  const [language, setLanguage] = useState(null);
  const [loading, setLoading] = useState(true);
  const [filterText, setFilterText] = useState("");
  const [filteredItems, setFilteredItems] = useState([]);
  const tableRef = useRef(null);

  const router = useRouter();

  useEffect(() => {
    const fetchSession = async () => {
      try {
        const sessionGet = await getSession();
        if (!sessionGet) {
          router.push("/login");
        } else {
          setSession(sessionGet.userData);
        }
      } catch (error) {
        console.error("Error al obtener la sesión", error);
        router.push("/login");
      }
    };

    fetchSession();
  }, []);

  useEffect(() => {
    if (!session) return;

    const fetchData = async () => {
      try {
        const data = await api.get(`/invitados/${session.id}`);

        const datatable = Array.isArray(data)
          ? data.map((item) => {
              const invitados = item.nombres
                .split(",")
                .map((nombre) => nombre.trim())
                .filter((nombre) => nombre !== "")
                .map((nombre, index, array) => {
                  if (index === array.length - 1) {
                    return nombre.charAt(0).toUpperCase() + nombre.slice(1);
                  }
                  return nombre;
                })
                .join(", ")
                .replace(/,([^,]*)$/, " y$1");

              return {
                nombres: invitados,
                asistentes: item.ctd === null ? 0 : item.ctd,
                adultos: item.adultos === null ? 0 : item.adultos,
                ninios: item.ninios === null ? 0 : item.ninios,
                telefono: item.telefono,
                obs: item.obs,
                estado:
                  item.estado === 1
                    ? "Confirmado"
                    : item.estado === 0
                    ? "Rechazado"
                    : "Pendiente",
              };
            })
          : [];

        setData(datatable);
        setLoading(false);
      } catch (error) {
        console.error("Error al cargar los datos:", error);
      } finally {
      }
    };

    fetchData();
  }, [session]);

  const handleFilter = (event) => {
    const filterValue = event.target.value;
    setFilterText(filterValue);

    const filteredData = data.filter((item) => {
      return Object.values(item)
        .join(" ")
        .toLowerCase()
        .includes(filterValue.toLowerCase());
    });

    if (filteredData.length === 0) {
      setFilteredItems([
        {
          mensaje: "No hay registro",
        },
      ]);
    } else {
      setFilteredItems(filteredData);
    }
  };

  const handleExportExcel = async () => {
    const workbook = new ExcelJS.Workbook();
    const worksheet = workbook.addWorksheet("Invitados");

    const dataToExport = filteredItems.length > 0 ? filteredItems : data;

    if (dataToExport.length === 0) {
      return;
    }

    worksheet.columns = Object.keys(dataToExport[0]).map((key) => ({
      header: key,
      key: key,
    }));

    dataToExport.forEach((row) => {
      worksheet.addRow(row);
    });

    const buffer = await workbook.xlsx.writeBuffer();
    const blob = new Blob([buffer], {
      type: "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
    });

    const link = document.createElement("a");
    link.href = URL.createObjectURL(blob);
    link.download = "Invitados.xlsx";
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };

  const handleExportPDF = () => {
    const dataToExport = filteredItems.length > 0 ? filteredItems : data;

    const doc = new jsPDF();

    doc.text("Lista de Invitados", 14, 15);

    const tableColumns = columns.map((col) => col.name);
    const tableRows = dataToExport.map((row) =>
      columns.map((col) =>
        typeof col.selector === "function"
          ? col.selector(row)
          : row[col.selector]
      )
    );

    autoTable(doc, {
      startY: 20,
      head: [tableColumns],
      body: tableRows,
    });

    doc.save("Invitados.pdf");
  };

  const handlePrint = () => {
    const dataToExport = filteredItems.length > 0 ? filteredItems : data;

    if (!dataToExport || dataToExport.length === 0) {
      console.error("No hay datos para imprimir.");
      return;
    }

    const printWindow = window.open();
    if (!printWindow) {
      console.error("No se pudo abrir la ventana de impresión.");
      return;
    }

    const styles = Array.from(document.styleSheets)
      .map((styleSheet) => {
        try {
          return styleSheet.cssRules
            ? Array.from(styleSheet.cssRules)
                .map((rule) => rule.cssText)
                .join("\n")
            : "";
        } catch (e) {
          return "";
        }
      })
      .join("\n");

    const tableHTML = `
      <table border="1" style="width:100%; border-collapse: collapse;">
        <thead>
          <tr>
          ${columns
            .map(
              (col) =>
                `<th style="padding: 8px; border: 1px solid #ddd; background: #f4f4f4;">${col.name}</th>`
            )
            .join("")}
         </tr>
        </thead>
        <tbody>
        ${dataToExport
          .map(
            (row) => `
          <tr>
            ${columns
              .map((col) => {
                const value =
                  typeof col.selector === "function"
                    ? col.selector(row)
                    : row[col.selector];
                return `<td style="padding: 8px; border: 1px solid #ddd;">${
                  value ?? ""
                }</td>`;
              })
              .join("")}
          </tr>
        `
          )
          .join("")}
        </tbody>
      </table>
    `;

    printWindow.document.write(`
      <html>
        <head>
          <title>Imprimir Tabla</title>
          <style>${styles}</style>
        </head>
        <body>
          <h1>Lista de Invitados</h1>
          ${tableHTML}
        </body>
      </html>
    `);

    printWindow.document.close();
    printWindow.focus();
    setTimeout(() => {
      printWindow.print();
      printWindow.close();
    }, 500);
  };

  useEffect(() => {
    if (!data) return null;

    fetch("/assets/lang/DataTables-ES.json")
      .then((res) => res.json())
      .then((data) => {
        setLanguage(data);
        setLoading(false);
      })
      .catch((error) => console.error("Error al cargar el idioma:", error));
  }, [data]);

  const totalInvitados = data.length;
  const confirmados = data.filter((i) => i.estado === "Confirmado").length;
  const rechazados = data.filter((i) => i.estado === "Rechazado").length;
  const pendientes = data.filter((i) => i.estado === "Pendiente").length;

  const getColumns = (usarSoloAsistentes, ocultarObs) => {
    const baseColumns = [
      {
        name: "Nombres",
        selector: (row) => row.nombres,
        sortable: true,
      },
      {
        name: "Telefono",
        selector: (row) => row.telefono,
        sortable: true,
      },
      {
        name: "Estado",
        selector: (row) => row.estado,
        sortable: true,
      },
    ];

    const asistentesColumn = {
      name: "Invitados",
      selector: (row) => row.asistentes,
      sortable: true,
    };

    const adultosColumn = {
      name: "Adultos",
      selector: (row) => row.adultos,
      sortable: true,
    };

    const niniosColumn = {
      name: "Niños",
      selector: (row) => row.ninios,
      sortable: true,
    };

    const obsColumn = {
      name: "Comentarios",
      selector: (row) => row.obs,
      sortable: true,
    };

    let columns = usarSoloAsistentes
      ? [baseColumns[0], asistentesColumn, ...baseColumns.slice(1)]
      : [baseColumns[0], adultosColumn, niniosColumn, ...baseColumns.slice(1)];

    if (!ocultarObs) {
      columns.push(obsColumn);
    }

    return columns;
  };

  const rows = filteredItems.length > 0 ? filteredItems : data;
  const usarSoloAsistentes = rows.every(
    (row) => parseInt(row.asistentes) === 1
  );
  const ocultarObs = rows.every((row) => row.obs === null || row.obs === "");
  const columns = getColumns(usarSoloAsistentes, ocultarObs);

  return (
    <Layout>
      {loading && !data && !session ? (
        <Loader />
      ) : (
        <>
          <div className="container vh-80 mt-6 mb-5">
            <div className="row">
              <div className="col-md-12 mt-5 mb-5">
                <h2 className="text-center">Lista de invitados</h2>
              </div>
            </div>
            {/*session?.cant_invitaciones > totalInvitados && (
              <div className="row">
                <div className="col-md-12 mb-5">
                  <Mantenedor_Invitados
                    id_usuario={session?.id}
                    id_invitado={0}
                  ></Mantenedor_Invitados>
                </div>
              </div>
            )*/}
            <div className="row">
              <div className="col-md-3">
                <div className="alert alert-info">
                  <div className="card-header">
                    <h5 className="card-title">Total Invitados</h5>
                  </div>
                  <div className="card-body mt-4 mb-2">
                    <p className="card-text fs-5 fw-bold ">{totalInvitados}</p>
                  </div>
                </div>
              </div>
              <div className="col-md-3">
                <div className="alert alert-success">
                  <div className="card-header">
                    <h5 className="card-title fw-bold ">Confirmados</h5>
                  </div>
                  <div className="card-body mt-4 mb-2">
                    <p className="card-text fs-5">{confirmados}</p>
                  </div>
                </div>
              </div>
              <div className="col-md-3">
                <div className="alert alert-danger">
                  <div className="card-header">
                    <h5 className="card-title">Rechazados</h5>
                  </div>
                  <div className="card-body mt-4 mb-2">
                    <p className="card-text fs-5">{rechazados}</p>
                  </div>
                </div>
              </div>
              <div className="col-md-3">
                <div className="alert alert-warning">
                  <div className="card-header">
                    <h5 className="card-title">Pendientes</h5>
                  </div>
                  <div className="card-body mt-4 mb-2">
                    <p className="card-text fs-5">{pendientes}</p>
                  </div>
                </div>
              </div>
            </div>

            <div className="row mt-5">
              <div className="col-md-6 mb-3"></div>
              <div className="col-md-4 mb-3">
                <input
                  type="text"
                  className="alert alert-warning form-control p-3 "
                  placeholder="Buscar"
                  value={filterText}
                  onChange={handleFilter}
                />
              </div>
              <div className="col-md-2 my-2">
                <button
                  className="btn btn-success me-2"
                  onClick={handleExportExcel}
                >
                  <FontAwesomeIcon icon={faFileExcel}></FontAwesomeIcon>
                </button>
                <button
                  className="btn btn-danger me-2"
                  onClick={handleExportPDF}
                >
                  <FontAwesomeIcon icon={faFilePdf}></FontAwesomeIcon>
                </button>
                <button className="btn btn-primary me-2" onClick={handlePrint}>
                  <FontAwesomeIcon icon={faPrint}></FontAwesomeIcon>
                </button>
              </div>
              <div ref={tableRef} className="col-md-12">
                <DataTable
                  columns={columns}
                  data={rows}
                  pagination
                  highlightOnHover
                  responsive
                  progressPending={data.length === 0}
                  paginationComponentOptions={language || {}}
                  id="dt_invitados"
                />
              </div>
            </div>
          </div>
        </>
      )}
    </Layout>
  );
}
