"use client";

import { useRouter } from "next/navigation";
import { useEffect, useState } from "react";
import Layout from "../layout/layout-page";
import Loader from "@/components/Loader";
import "@/styles/custom-intranet.css";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { faPencil, faSave } from "@fortawesome/free-solid-svg-icons";
import { api } from "@/libs/apiClient";
import { getSession } from "@/libs/session";

export default function Intranet() {
  const [session, setSession] = useState(undefined);
  const [editando, setEditando] = useState(false);
  const [password, setPassword] = useState("");
  const [password2, setPassword2] = useState("");
  const [errorMail, setErrorMail] = useState(null);
  const [errorPass, setErrorPass] = useState(null);
  const [loading, setLoading] = useState(null);
  const [resultOk, setResultOk] = useState(null);
  const [resultErr, setResultErr] = useState(null);
  const [showAlert, setShowAlert] = useState(true);

  const router = useRouter();

  useEffect(() => {
    const fetchSession = async () => {
      try {
        const session = await getSession();
        if (!session) {
          router.push("/login");
        } else {
          setSession(session.userData);
        }
      } catch (error) {
        console.error("Error al obtener la sesión", error);
        router.push("/login");
      }
    };

    fetchSession();
  }, []);

  const handleSubmit = async () => {
    let validaForm = true;

    if (
      (password === "" && password2 !== "") ||
      (password !== "" && password2 === "")
    ) {
      setErrorPass("Debe ingresar ambas contraseñas o dejarlas en blanco.");
      validaForm = false;
    } else if (password !== "" && password2 !== "" && password !== password2) {
      setErrorPass("Las contraseñas no coinciden.");
      setPassword("");
      setPassword2("");
      validaForm = false;
    } else {
      setErrorPass(null);
    }

    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(session.email)) {
      setErrorMail("Ingresar un Email válido.");
      validaForm = false;
    } else {
      setErrorMail(null);
    }

    if (!validaForm) return;

    setLoading(true);
    try {
      const data = await api.put("/usuarios/updateUsuario", {
        nombres: session.nombres,
        user: session.usuario,
        email: session.email,
        pass: password,
      });

      if (data) {
        setResultOk(data.message);

        const resSession = await getSession();

        const menuData = resSession.menuData;
        const userData = data.userData;

        const session = await api.post("/session/create", {
          userData,
          menuData,
        });
      } else {
        setResultErr(data.message);
      }
    } catch (error) {
      setResultErr(error.message);
    } finally {
      setLoading(false);
      setEditando(false);
    }
  };

  const handleChange = (e) => {
    const { name, value } = e.target;
    if (name === "password") {
      setPassword(value);
    } else if (name === "password2") {
      setPassword2(value);
    } else {
      setSession((prevSession) => ({
        ...prevSession,
        [name]: value,
      }));
    }
  };

  const habilitarEdicion = () => {
    setEditando(true);
  };

  return (
    <Layout>
      {session === undefined || loading ? (
        <Loader></Loader>
      ) : (
        <div className="mt-6 mb-5 vh-80">
          <div className="container">
            <div className="row col-md-12">
              <div className="col-md-4">
                <div className="card mb-4 mb-xl-0">
                  <div className="card-header">
                    <h5 className="fw-bold text-uppercase text-center mt-2 mb-2">
                      Bievenido {session.usuario}
                    </h5>
                  </div>
                  <div className="card-body text-center">
                    <img
                      className="img-account-profile background-color-grey rounded-circle mb-2"
                      src="/assets/images/banners/banner_cuadrado.png"
                      width={290}
                      height={290}
                      alt=""
                    />
                  </div>
                </div>
              </div>
              <div className="col-md-8">
                <div className="card mb-4">
                  <div className="card-header">
                    <h5 className="fw-bold text-uppercase mt-2 mb-2">
                      Mis Datos
                    </h5>
                  </div>
                  <div className="card-body">
                    {resultOk && showAlert && (
                      <div className="alert alert-success" role="alert">
                        {resultOk}
                        <button
                          type="button"
                          className="close"
                          data-dismiss="alert"
                          aria-label="Close"
                          onClick={() => setShowAlert(false)}
                        >
                          <span aria-hidden="true">&times;</span>
                        </button>
                      </div>
                    )}
                    {resultErr && showAlert && (
                      <div className="alert alert-danger" role="alert">
                        {resultErr}
                        <button
                          type="button"
                          className="close"
                          data-dismiss="alert"
                          aria-label="Close"
                          onClick={() => setShowAlert(false)}
                        >
                          <span aria-hidden="true">&times;</span>
                        </button>
                      </div>
                    )}
                    <form className="mt-4 mb-4" autoComplete="off">
                      <div className="row col-md-12">
                        <div className="col-md-6">
                          <label className="fw-bold">Nombres</label>
                          <div className="form-group">
                            <input
                              type="text"
                              className="form-control"
                              name="nombres"
                              id="nombres"
                              value={session.nombres}
                              onChange={handleChange}
                              placeholder="Nombres"
                              disabled={!editando}
                            />
                          </div>
                        </div>
                        <div className="col-md-4">
                          <label className="fw-bold">Teléfono</label>
                          <div className="form-group">
                            <input
                              type="text"
                              className="form-control"
                              name="telefono"
                              id="telefono"
                              value={session.telefono ?? ""}
                              placeholder="Telefono"
                              disabled
                            />
                          </div>
                        </div>
                        <div className="col-md-4">
                          <label className="fw-bold">Email</label>
                          <div className="form-group">
                            <input
                              type="email"
                              className="form-control"
                              name="email"
                              id="email"
                              value={session.email ?? ""}
                              onChange={handleChange}
                              placeholder="Email"
                              disabled={!editando}
                            />
                          </div>
                          {errorMail && (
                            <span
                              style={{ display: "block" }}
                              className="invalid-feedback mb-2"
                            >
                              <strong>{errorMail}</strong>
                            </span>
                          )}
                        </div>
                        <div className="col-md-4 d-none">
                          <label className="fw-bold">Paquete</label>
                          <div className="form-group">
                            <input
                              type="text"
                              className="form-control"
                              name="paquete"
                              id="paquete"
                              value={session.id_plan ?? ""}
                              placeholder="Paquete"
                              disabled
                            />
                          </div>
                        </div>
                        <div className="col-md-4 d-none">
                          <label className="fw-bold">Diseño</label>
                          <div className="form-group">
                            <input
                              type="text"
                              className="form-control"
                              name="designe"
                              id="designe"
                              value={session.id_template ?? ""}
                              placeholder="Diseño"
                              disabled
                            />
                          </div>
                        </div>
                        <div className="col-md-4">
                          <label className="fw-bold">Contraseña</label>
                          <div className="form-group">
                            <input
                              type="password"
                              className="form-control"
                              name="password"
                              id="password"
                              value={password}
                              onChange={handleChange}
                              placeholder="Contraseña"
                              disabled={!editando}
                            />
                          </div>
                        </div>
                        <div className="col-md-4">
                          <label className="fw-bold">
                            Re-ingresar contraña
                          </label>
                          <div className="form-group">
                            <input
                              type="password"
                              className="form-control"
                              name="password2"
                              id="password2"
                              value={password2}
                              onChange={handleChange}
                              placeholder="Re-ingresar Contraseña"
                              disabled={!editando}
                            />
                          </div>
                        </div>
                        {errorPass && (
                          <span
                            style={{ display: "block" }}
                            className="invalid-feedback"
                          >
                            <strong>{errorPass}</strong>
                          </span>
                        )}
                      </div>
                    </form>
                    <div className="col-md-12 mt-4">
                      <div className="form-group">
                        {!editando ? (
                          <button
                            className="btn btn-primary"
                            type="button"
                            onClick={habilitarEdicion}
                          >
                            <FontAwesomeIcon icon={faPencil}></FontAwesomeIcon>{" "}
                            Modificar mis datos
                          </button>
                        ) : (
                          <button
                            className="btn btn-success"
                            type="button"
                            onClick={handleSubmit}
                          >
                            <FontAwesomeIcon icon={faSave}></FontAwesomeIcon>{" "}
                            Guardar
                          </button>
                        )}
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}
    </Layout>
  );
}
